### ARGUMENTS ##################################################################

param (
    [parameter(Position=0,Mandatory=$true)]
    [ValidateSet("shared","vps")]
    [string]$InstallType = "shared",

    [parameter(Position=1,Mandatory=$true)]
    [ValidateSet("all","7.4","8.0","8.1","8.2", "8.3")]
    [string]$InstallVersion = "all"
)


### FUNCTIONS ##################################################################

# Check elevation
function Test-Administrator
{
    $CurrentUser = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
    $CurrentUser.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# Initialize path
function Init-Path
{
    param(
        [Parameter(Mandatory)][string]$Path
    )
    Write-Host "`nInitializing $($Path) directory"
    if (!(Test-Path -PathType Container $Path)) {
        New-Item -ItemType Directory -Force -Path $Path | Out-Null
    }
}

# Download file
function Get-File
{
    param(
        [Parameter(Mandatory)][string]$FileSource,
        [Parameter(Mandatory)][string]$FileDestination
    )
    Write-Host "`nDownloading to $($FileDestination)"
    Invoke-WebRequest -Uri $FileSource -UseBasicParsing -OutFile $FileDestination
    if (!(Test-Path $FileDestination)) {
        Write-Host "Downloading $FileDestination failed" -Foregroundcolor Red
        Exit
    } else {
        Write-Host "Download complete" -ForegroundColor Green
    }
}

# Extract file
function Unzip-File
{
    param($UnzipSource, $UnzipDestination)
    Write-Host "`nExtracting $($UnzipSource) to $($UnzipDestination)"
    7zip x $UnzipSource -y -o"$($UnzipDestination)" | Out-Null
    Write-Host "Extraction complete" -ForegroundColor Green
}

# Install file
function Install-File
{
    param($InstallerSource)
    try {
        Write-Host "`nInstalling $InstallerSource"
        $proc = Start-Process -FilePath $InstallerSource -ArgumentList "/quiet /norestart" -Wait -PassThru
        $proc.waitForExit()
        Write-Host "Installation complete" -ForegroundColor Green
    } catch [expression] {
        Write-Host "Installation failed" -ForegroundColor Red
    }
}


### INITIALIZE #################################################################

[System.Net.ServicePointManager]::SecurityProtocol = 'Tls,Tls11,Tls12';

# Check privileges
if (!(Test-Administrator)) {
    Write-Host "This script requires elevation (Run as Administrator)" -ForegroundColor Red
    Exit
}

# Define install version
$PhpVersions = @{
    "shared" = @(
        @{
            Name    = "windows-shared-version-7.4"
            Version = "7.4"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/367/repository/archive.zip?private_token=glpat-oKzJ8mwb8WLGx1PKghCp"
        }
        @{
            Name    = "windows-shared-version-8.0"
            Version = "8.0"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/374/repository/archive.zip?private_token=glpat-k8fMrnJXyfssAwJwwyve"
        }
        @{
            Name    = "windows-shared-version-8.1"
            Version = "8.1"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/369/repository/archive.zip?private_token=glpat-K-9smp-zAnsPh8z6V34G"
        }
        @{
            Name    = "windows-shared-version-8.2"
            Version = "8.2"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/366/repository/archive.zip?private_token=glpat-D55X2tgZNgp3WkeKY_t2"
        }
        @{
            Name    = "windows-shared-version-8.3"
            Version = "8.3"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/377/repository/archive.zip?private_token=glpat-fGWy2TCgt3zxdsc9JUZx"
        }
    )
    "vps" = @(
        @{
            Name    = "windows-vps-version-7.4"
            Version = "7.4"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/370/repository/archive.zip?private_token=glpat-5HEGpD2F7pp4D6c2_a9Q"
        }
        @{
            Name    = "windows-vps-version-8.0"
            Version = "8.0"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/371/repository/archive.zip?private_token=glpat-xbJ7ryZacb7xYCPG3XNf"
        }
        @{
            Name    = "windows-vps-version-8.1"
            Version = "8.1"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/372/repository/archive.zip?private_token=glpat-omsZ2iHGVEiWqU4WYyQT"
        }
        @{
            Name    = "windows-vps-version-8.2"
            Version = "8.2"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/373/repository/archive.zip?private_token=glpat-7YKRuZ1xGyzH7PYeeW1s"
        }
        @{
            Name    = "windows-vps-version-8.3"
            Version = "8.3"
            Source  = "https://gitlab1.hostek.com/api/v4/projects/378/repository/archive.zip?private_token=glpat-TLJzGa3iYw-4tqFn-Nd3"
        }
    )
}

# Define common files
$PhpTests = @{
    Name   = "tests"
    Source = "https://gitlab1.hostek.com/api/v4/projects/375/repository/archive.zip?private_token=glpat-sRwaDQeVJYe2As1ydaLN"
}

# Define install path
$InstallDirectory = if ($InstallType -eq "shared") {"D:\PHP"} else {"C:\PHP"}

# Define date
$CurrentTimestamp = $(get-date -f yyMMdd_hhmmss)


### INSTALL 7-ZIP ##############################################################

$ZipSource      = 'https://7-zip.org/a/7z2301-x64.exe'
$ZipDestination = Join-Path $env:TEMP '7z2301-x64.msi'
Get-File $ZipSource $ZipDestination
Install-File $ZipDestination

# Alias application
Set-Alias -scope "global" -name "7zip" -value "$env:ProgramFiles\7-Zip\7z.exe"


### INSTALL VCREDIST ###########################################################

$RuntimeSource        = "https://aka.ms/vs/17/release/vc_redist.x86.exe"
$RuntimeDestination   = Join-Path $env:TEMP "vcredist_x86.exe"
Get-File $RuntimeSource $RuntimeDestination
Install-File $RuntimeDestination

$Runtime64Source      = "https://aka.ms/vs/17/release/vc_redist.x64.exe"
$Runtime64Destination = Join-Path $env:TEMP "vcredist_x64.exe"
Get-File $RuntimeSource $RuntimeDestination
Install-File $RuntimeDestination


### INSTALL PHP ################################################################

# Initialize install directory
Init-Path $InstallDirectory

# Perform install
foreach($Version in $PhpVersions.$InstallType) {
    if (($InstallVersion -eq "all") -or ($InstallVersion -eq $Version.Version)) {

        # Alias paths
        $OldPath      = Join-Path $InstallDirectory $Version.Version
        $NewPath      = Join-Path $InstallDirectory "$($Version.Version)_BACKUP_$($CurrentTimestamp)"
        $DownloadPath = Join-Path $InstallDirectory "$($Version.Name).zip"

        # Backup old installs
        if (Test-Path -PathType Container $OldPath) {
            Write-Host "`nArchiving $($Version.Version)"
            Rename-Item $OldPath $NewPath
        }

        # Download new version
        Get-File $Version.Source $DownloadPath

        # Extract new version
        Unzip-File $DownloadPath $InstallDirectory

        # Rename directory
        Write-Host "`nRenaming $($Version.Version) directory"
        $RenameDirectory = (Get-ChildItem -Path $InstallDirectory -Filter "$($Version.Name)*" -Directory).FullName
        Rename-Item -Path $RenameDirectory -NewName $Version.Version
        Write-Host "Rename complete" -ForegroundColor Green

        # Remove download
        Write-Host "`nDeleting $($Version.Name).zip"
        Remove-Item $DownloadPath
    }
}


### TEMP #######################################################################

# Create temp directory
Init-Path 'C:\temp\php'


### LOGS #######################################################################

# Define path
$LogsOldPath = Join-Path $InstallDirectory 'logs'
$LogsNewPath = Join-Path $InstallDirectory "logs_BACKUP_$($CurrentTimestamp)"

# Backup directory
if (Test-Path -PathType Container $LogsOldPath) {
    Write-Host "`nArchiving logs"
    Rename-Item $LogsOldPath $LogsNewPath
}

# Initialize path
Init-Path $LogsOldPath

# Create mail log
$MailLogsPath = Join-Path $LogsOldPath "mail.log"
New-Item $MailLogsPath -ItemType File | Out-Null


### TESTS ######################################################################

# Delete tests
$TestsPath = Join-Path $InstallDirectory "tests"
if (Test-Path -PathType Container $TestsPath) {
    Write-Host "`nDeleting tests"
    Remove-Item -Force -Recurse $TestsPath
}

# Download tests
$TestsSource      = $PhpTests.Source
$TestsDestination = Join-Path $InstallDirectory "tests.zip"
Get-File $TestsSource $TestsDestination

# Extract tests
Unzip-File $TestsDestination $InstallDirectory
$OldPath = (Get-ChildItem -Path $InstallDirectory -Filter "$($Tests.Name)*" -Directory).FullName
$NewPath = $PhpTests.Name

# Rename directory
Write-Host "`nRenaming tests directory"
$RenameDirectory = (Get-ChildItem -Path $InstallDirectory -Filter "tests-master*" -Directory).FullName
Rename-Item -Path $RenameDirectory -NewName $PhpTests.Name
Write-Host "Rename complete" -ForegroundColor Green

# Remove download
Remove-Item $TestsDestination


### CERTIFICATE STORE ##########################################################

# Download certificate
$CertificateSource      = "https://curl.se/ca/cacert.pem"
$CertificateDestination = Join-Path $InstallDirectory "cacert.pem"
Get-File $CertificateSource $CertificateDestination


### FINALIZE ###################################################################

Write-Host "`nPHP installation complete!" -ForegroundColor Blue
Write-Host "That's all folks..." -ForegroundColor Yellow
